{ config, lib, pkgs, ... }: 
# ''${lib.getExe' pkgs.pipewire "pw-cat"} --media-role Notification -p "${pkgs.kdePackages.oxygen-sounds}/share/sounds/oxygen/stereo/desktop-login-long.ogg" &''
let 
  cfgPath = [ "wayland" "startupSound" ];

  cfg = lib.getAttrFromPath cfgPath config;
in
{
  options = lib.setAttrByPath cfgPath {
    enable = lib.mkEnableOption "wayland session startup sound";
    sound = lib.mkOption {
      type = lib.types.path;

      default = /${pkgs.kdePackages.oxygen-sounds}/share/sounds/Oxygen-Sys-Log-In-Long.ogg;

      defaultText = lib.literalExpression "/\${pkgs.kdePackages.oxygen-sounds}/share/sounds/Oxygen-Sys-Log-In-Long.ogg";
      example = lib.literalExpression "/\${pkgs.mint-artwork}/share/sounds/linuxmint-login.wav";

      description = "Path to sound to play on wayland session login.";
    };

    audioBackend = lib.mkOption {
      type = lib.types.enum [ "ffmpeg" "pipewire" "pulseaudio" ];

      default = "ffmpeg";
      example = "pipewire";

      description = ''
        Audio backend to use to play startup audio sound.

        `ffmpeg` - use ffplay
        `pipewire` - use pw-cat (note that only this backend supports media roles
        `pulseaudio - use pacat
      '';
    };

    pipewireMediaRole = lib.mkOption {
      type = lib.types.string;

      default = "Notification";
      example = "Music";

      description = "Media role to use with pw-cat, check https://docs.pipewire.org/page_man_pipewire-props_7.html#node-prop__media_role for available roles.";
    };

    volume = lib.mkOption {
      type = lib.types.float;

      default = 1.00;
      example = 0.70;

      description = "Volume of startup sound, 0.00 - 0%, 0.56 - 56%, 1.00 - 100%.";
    };
  };

  config = lib.mkIf cfg.enable {
    systemd.user.services.wayland-startup-sound = {
      Unit = {
        Description = "Wayland startup sound";
        WantedBy = [ config.wayland.systemd.target ];
        After = [ config.wayland.systemd.target ] ++ 
          (lib.optional (cfg.audioBackend == "pipewire") "wireplumber.service") ++
          (lib.optional (cfg.audioBackend == "pulseaudio") "pulseaudio.service");
        Requires = 
          (lib.optional (cfg.audioBackend == "pipewire") "wireplumber.service") ++
          (lib.optional (cfg.audioBackend == "pulseaudio") "pulseaudio.service");
      };

      Service = {
        Type = "oneshot";
        Restart = "no";
        ExecStart = 
          if cfg.audioBackend == "ffmpeg" then
            "${lib.getExe' pkgs.ffmpeg "ffplay"} -autoexit -volume ${lib.toString (builtins.ceil (cfg.volume * 100))} ${lib.escapeShellArg (lib.toString cfg.sound)}"
          else if cfg.audioBackend == "pipewire" then
            "${lib.getExe' pkgs.pipewire "pw-cat"} --media-role '${cfg.pipewireMediaRole}' --volume ${toString cfg.volume} -p ${lib.escapeShellArg (lib.toString cfg.sound)}"
          else if cfg.audioBackend == "pulseaudio" then
            "${lib.getExe' pkgs.pulseaudio "pacat"} --volume=${lib.toString (builtins.ceil (cfg.volume * 65536))} --client-name=wayland-session-startup-sound --stream-name=wayland-session-startup-sound ${lib.escapeShellArg (lib.toString cfg.sound)}"
          else throw "Not supported value ${cfg.audiobackend} in ${lib.concatStringsSep "." cfgPath} module.";
      };
    };
  };
}
